﻿using System;
using System.Globalization;
using System.Windows.Forms;
using Psion.RFID.HF;

namespace HF_Demo
{
    public partial class EepromForm : Form
    {
        private readonly Reader _reader;

        public Eeprom[] EepromElements = new []{

            new Eeprom {Description = "Station ID", RegisterValue = 0x04, Value = 0, DefaultValue = 0x04, BitDetails = false}, 
            
            new Eeprom {Description = "Protocol Configuration", RegisterValue = 0x05, Value = 0, DefaultValue = 0xDF, BitDetails = true,
            BitDesc = new[]
                          {
#if WAP4
                              "Autostart","Protocol mode",
#else
                              "","",                              
#endif
                              "Extended ID","", "","","",""
                          }}, 

#if WAP4
            new Eeprom {Description = "Baud Rate Configuration", RegisterValue = 0x0C, Value = 0, DefaultValue = 0x04, BitDetails = true,
            BitDesc = new[]{"BS0","BS1","BS2","", "","","",""}},  
#endif

            new Eeprom {Description = "Operation Mode", RegisterValue = 0x0E, Value = 0, DefaultValue = 0xFF, BitDetails = true,
            BitDesc = new[]{"14443A","14443B","SR716","Felica", "ISO15693","","ICODEUID","GTML"}},  
            
            new Eeprom {Description = "Reset Off Time", RegisterValue = 0x14, Value = 0, DefaultValue = 0x03}, 
            
            new Eeprom {Description = "Reset Recovery Time", RegisterValue = 0x15, Value = 0, DefaultValue = 0x0F}, 
            
            new Eeprom {Description = "Rx Threshold", RegisterValue = 0x1E, Value = 0, DefaultValue = 0xA5, BitDetails = true,
            BitDesc = new[]{"Min3","Min2","Min1","Min0", "Coll3","Coll2","Coll1","Coll0"}},  

#if WAP4
            new Eeprom {Description = "Hardware Configuration", RegisterValue = 0x20, Value = 0, DefaultValue = 0xF8, BitDetails = true,
            BitDesc = new[]{"SAM","Level Shifter","","", "","","",""}},    
#endif
            
            new Eeprom {Description = "Modulation Index", RegisterValue = 0x21, Value = 0, DefaultValue = 0x1A,  BitDetails = true,
            BitDesc = new[]{"ModReg0","ModReg1","ModReg2","ModReg3", "ModReg4","ModReg5","",""}}, 
            
            new Eeprom {Description = "RF Level", RegisterValue = 0x22, Value = 0, DefaultValue = 0x76, BitDetails = true,
            BitDesc = new[]{"RFLevel0","RFLevel1","RFLevel2","RFLevel3", "RxGain0","RxGain1","RxGain2",""}},  

#if WAP4            
            new Eeprom {Description = "Protocol Configuration 2", RegisterValue = 0x28, Value = 0, DefaultValue = 0xFF, BitDetails = true,
            BitDesc = new[]{"Read after Write","Verify data","","", "","","",""}},
#endif
            
            new Eeprom {Description = "PICC Timeout", RegisterValue = 0x29, Value = 0, DefaultValue = 0x30, BitDetails = true,
            BitDesc = new[]{"PICC Timeout0","PICC Timeout1","PICC Timeout2","PICC Timeout3", "PICC Timeout4","PICC Timeout5","PICC Timeout6","PICC Timeout7"}},

            new Eeprom {Description = "Cw Amplitude", RegisterValue = 0x2A, Value = 0, DefaultValue = 0x00, BitDetails = true,
            BitDesc = new[]{"cw_amp bit0","cw_amp bit1","","", "","","",""}},

            new Eeprom {Description = "Cw Max", RegisterValue = 0x2B, Value = 0, DefaultValue = 0x00, BitDetails = true,
            BitDesc = new[]{"cwmax","","","", "","","",""}},

            new Eeprom {Description = "TxiLoad", RegisterValue = 0x2C, Value = 0, DefaultValue = 0x0A, BitDetails = true,
            BitDesc = new[]{"Tx_iLoad bit0","Tx_iLoad bit1","Tx_iLoad bit2","Tx_iLoad bit3", "","","",""}},
        };

        private bool _wasUpdated;

        public EepromForm(Reader reader)
        {
            InitializeComponent();

            if (!Device.IsWM) mainMenu1.Dispose();

            _reader = reader;

        }

        private void ReadEeprom()
        {
            comboBoxEepromElement.Items.Clear();
            foreach (Eeprom eepromElement in EepromElements)
            {
                var data = _reader.ReadEeprom(eepromElement.RegisterValue);

                if (data == null || data.Length == 0)
                    throw new Exception("Error getting Eeprom value");

                eepromElement.Value = data[0];

                comboBoxEepromElement.Items.Add(eepromElement.Description);
            }
        }

        private void WriteEeprom(Eeprom eeprom)
        {
            var data = _reader.WriteEeprom(eeprom.RegisterValue, eeprom.Value);

            if (data == null || data.Length == 0)
                throw new Exception("Error writing Eeprom value");
        }


        private void ComboBoxEepromElementSelectedIndexChanged(object sender, EventArgs e)
        {
            panelBits.Visible = false;
            
            if (comboBoxEepromElement.SelectedIndex != -1)
            {
                Eeprom eeprom = EepromElements[comboBoxEepromElement.SelectedIndex];
                labelRegisterValue.Text =
                    string.Format("0x{0:X2}",
                                 eeprom.RegisterValue);
                textBoxValue.Text =
                    string.Format("{0:X2}",
                                 eeprom.Value);

                if (eeprom.BitDetails)
                {
                    UpdateBitDetails(eeprom);


                    panelBits.Visible = true;
                    
                }

                textBoxDefaultValue.Text =
                    string.Format("{0:X2}",
                 eeprom.DefaultValue);

            }
        }

        private void UpdateBitDetails(Eeprom eeprom)
        {
            //var bitValues = new bool[8];
            bool [] bitValues =GetBits(eeprom.Value);
            int i = 0;
            foreach (string bitDesc in eeprom.BitDesc)
            {
                CheckBox checkBox = null;
                switch (i)
                {
                    case 0: checkBox = checkBoxBit0; break;
                    case 1: checkBox = checkBoxBit1; break;
                    case 2: checkBox = checkBoxBit2; break;
                    case 3: checkBox = checkBoxBit3; break;
                    case 4: checkBox = checkBoxBit4; break;
                    case 5: checkBox = checkBoxBit5; break;
                    case 6: checkBox = checkBoxBit6; break;
                    case 7: checkBox = checkBoxBit7; break;
                }
                if (checkBox != null)
                {
                    if (bitDesc == string.Empty)
                    {
                        checkBox.Enabled = false;
                        checkBox.Text = "RFU";
                    }
                    else
                    {
                        checkBox.Enabled = true;
                        checkBox.Text = bitDesc;
                    }
                    checkBox.Checked = bitValues[i];
                }

                i++;
            }
        }

        private bool[] GetBits(byte value)
        {
            string s = Convert.ToString(value, 2).PadLeft(8, '0');
            var data = new bool[8];
            for (int i = 0; i < s.Length; i++)
            {
                data[7-i] = Convert.ToBoolean(Convert.ToInt32(s[i].ToString(CultureInfo.InvariantCulture), 2));
            }
            return data;
        }

        private void EepromFormLoad(object sender, EventArgs e)
        {
            try
            {
                ReadEeprom();

                if (comboBoxEepromElement.Items.Count != 0)
                    comboBoxEepromElement.SelectedIndex = 0;

            }
            catch (Exception exception)
            {
                MessageBox.Show(exception.Message, "Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation,
                                MessageBoxDefaultButton.Button1);
                Close();
            }
        }

        private void TextBoxValueTextChanged(object sender, EventArgs e)
        {
            try
            {
                byte data = Convert.ToByte(textBoxValue.Text, 16);

                // check if it is hexadecimal value
                if (comboBoxEepromElement.SelectedIndex != -1)
                {

                    Eeprom eeprom = EepromElements[comboBoxEepromElement.SelectedIndex];
                    eeprom.Value = data;
                    UpdateBitDetails(eeprom);
                }
            }
            catch (Exception exception)
            {
                MessageBox.Show(exception.Message, "Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation,
                                MessageBoxDefaultButton.Button1);
            }


        }

        private void ButtonApplyClick(object sender, EventArgs e)
        {
            //
            if (comboBoxEepromElement.SelectedIndex != -1)
            {
                try
                {
                    Eeprom eeprom = EepromElements[comboBoxEepromElement.SelectedIndex];
                    WriteEeprom(eeprom);
                    MessageBox.Show("New setting applied", "Information");
                    _wasUpdated = true;

                }
                catch (Exception exception)
                {
                    MessageBox.Show(exception.Message, "Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation,
                                    MessageBoxDefaultButton.Button1);
                }
            }
        }

        private void EepromFormClosing(object sender, System.ComponentModel.CancelEventArgs e)
        {
            try
            {
                // Reset the reader to get the parameters active
                if (_wasUpdated)
                {
                    _reader.Reset();
                    MessageBox.Show("Eeprom successfully modified", "Information");
                }
                //else
                //    MessageBox.Show("No modification performed on Eeprom", "Information");
            }
            catch (Exception)
            {

            }

        }

        private void UpdateByte()
        {
            string bitStr =
                (checkBoxBit7.Checked ? "1" : "0") +
                (checkBoxBit6.Checked ? "1" : "0") +
                (checkBoxBit5.Checked ? "1" : "0") +
                (checkBoxBit4.Checked ? "1" : "0") +
                (checkBoxBit3.Checked ? "1" : "0") +
                (checkBoxBit2.Checked ? "1" : "0") +
                (checkBoxBit1.Checked ? "1" : "0") +
                (checkBoxBit0.Checked ? "1" : "0");
            int value = Convert.ToInt32(bitStr, 2);
            if (comboBoxEepromElement.SelectedIndex != -1)
            {
                Eeprom eeprom = EepromElements[comboBoxEepromElement.SelectedIndex];
                textBoxValue.Text = string.Format("{0:X2}", value);
                eeprom.Value = (byte) value;
            }
        }

        private void CheckBoxBit7CheckStateChanged(object sender, EventArgs e)
        {
            UpdateByte();
        }

        private void CheckBoxBit6CheckStateChanged(object sender, EventArgs e)
        {
            UpdateByte();
        }

        private void CheckBoxBit5CheckStateChanged(object sender, EventArgs e)
        {
            UpdateByte();
        }

        private void CheckBoxBit4CheckStateChanged(object sender, EventArgs e)
        {
            UpdateByte();
        }

        private void CheckBoxBit3CheckStateChanged(object sender, EventArgs e)
        {
            UpdateByte();
        }

        private void CheckBoxBit2CheckStateChanged(object sender, EventArgs e)
        {
            UpdateByte();
        }

        private void CheckBoxBit1CheckStateChanged(object sender, EventArgs e)
        {
            UpdateByte();
        }

        private void CheckBoxBit0CheckStateChanged(object sender, EventArgs e)
        {
            UpdateByte();
        }

        private void ButtonReloadClick(object sender, EventArgs e)
        {
            try
            {
                textBoxValue.Text = textBoxDefaultValue.Text;

            }
            catch (Exception exception)
            {
                MessageBox.Show(exception.Message, "Error", MessageBoxButtons.OK, MessageBoxIcon.Exclamation,
                                MessageBoxDefaultButton.Button1);
            }
        }

        private void MenuItem1Click(object sender, EventArgs e)
        {
            Close();
        }
    }
}